/**
  ******************************************************************************
  * @file    usb1_interface.c
  * @author  MCD Application Team
  * @brief   Contains USB HW configuration
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "interface_conf.h"
#include "platform.h"
#include "usbd_core.h"
#include "usbd_hid.h"
#include "iwdg_interface.h"
#include "wwdg_interface.h"
#include "openbl_usart_cmd.h"
#include "openbootloader_conf.h"

#if IO_INTERFACE_USB1

/*!< hidraw in endpoint */
#define HIDRAW_IN_EP           0x81
#define HIDRAW_IN_SIZE         64
#define HIDRAW_IN_INTERVAL     1

/*!< hidraw out endpoint */
#define HIDRAW_OUT_EP          0x02
#define HIDRAW_OUT_EP_SIZE     64
#define HIDRAW_OUT_EP_INTERVAL 1

/*Puya 0x50 0x75 0x79 0x61*/
#define USBD_VID               0x36B7
#define USBD_PID               (((DEVICE_ID_MSB & 0xFF) << 8 ) | (DEVICE_ID_LSB & 0xFF))
#define USBD_MAX_POWER         100
#define USBD_LANGID_STRING     1033

/*!< config descriptor size */
#define USB_HID_CONFIG_DESC_SIZ (9 + 9 + 9 + 7 + 7)

/*!< custom hid report descriptor size */
#define HID_CUSTOM_REPORT_DESC_SIZE 34

/*!< global descriptor */
static const uint8_t hid_descriptor[] =
{
  USB_DEVICE_DESCRIPTOR_INIT(USB_2_0, 0x00, 0x00, 0x00, USBD_VID, USBD_PID, 0x0002, 0x01),
  USB_CONFIG_DESCRIPTOR_INIT(USB_HID_CONFIG_DESC_SIZ, 0x01, 0x01, USB_CONFIG_BUS_POWERED, USBD_MAX_POWER),
  /************** Descriptor of Custom interface *****************/
  0x09,                          /* bLength: Interface Descriptor size */
  USB_DESCRIPTOR_TYPE_INTERFACE, /* bDescriptorType: Interface descriptor type */
  0x00,                          /* bInterfaceNumber: Number of Interface */
  0x00,                          /* bAlternateSetting: Alternate setting */
  0x02,                          /* bNumEndpoints */
  0x03,                          /* bInterfaceClass: HID */
  0x01,                          /* bInterfaceSubClass : 1=BOOT, 0=no boot */
  0x00,                          /* nInterfaceProtocol : 0=none, 1=keyboard, 2=mouse */
  0,                             /* iInterface: Index of string descriptor */
  /******************** Descriptor of Custom HID ********************/
  0x09,                    /* bLength: HID Descriptor size */
  HID_DESCRIPTOR_TYPE_HID, /* bDescriptorType: HID */
  0x11,                    /* bcdHID: HID Class Spec release number */
  0x01,
  0x00,                        /* bCountryCode: Hardware target country */
  0x01,                        /* bNumDescriptors: Number of HID class descriptors to follow */
  0x22,                        /* bDescriptorType */
  HID_CUSTOM_REPORT_DESC_SIZE, /* wItemLength: Total length of Report descriptor */
  0x00,
  /******************** Descriptor of Custom in endpoint ********************/
  0x07,                         /* bLength: Endpoint Descriptor size */
  USB_DESCRIPTOR_TYPE_ENDPOINT, /* bDescriptorType: */
  HIDRAW_IN_EP,                 /* bEndpointAddress: Endpoint Address (IN) */
  0x03,                         /* bmAttributes: Interrupt endpoint */
  WBVAL(HIDRAW_IN_SIZE),        /* wMaxPacketSize: 4 Byte max */
  HIDRAW_IN_INTERVAL,           /* bInterval: Polling Interval */
  /******************** Descriptor of Custom out endpoint ********************/
  0x07,                         /* bLength: Endpoint Descriptor size */
  USB_DESCRIPTOR_TYPE_ENDPOINT, /* bDescriptorType: */
  HIDRAW_OUT_EP,                /* bEndpointAddress: Endpoint Address (IN) */
  0x03,                         /* bmAttributes: Interrupt endpoint */
  WBVAL(HIDRAW_OUT_EP_SIZE),    /* wMaxPacketSize: 4 Byte max */
  HIDRAW_OUT_EP_INTERVAL,       /* bInterval: Polling Interval */
  /* 73 */
  ///////////////////////////////////////
  /// string0 descriptor
  ///////////////////////////////////////
  USB_LANGID_INIT(USBD_LANGID_STRING),
  ///////////////////////////////////////
  /// string1 descriptor
  ///////////////////////////////////////
  0x08,                       /* bLength */
  USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
  'A', 0x00,                  /* wcChar0 */
  'R', 0x00,                  /* wcChar1 */
  'M', 0x00,                  /* wcChar2 */
  ///////////////////////////////////////
  /// string2 descriptor
  ///////////////////////////////////////
  0x2E,                       /* bLength */
  USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
  'A', 0x00,                  /* wcChar0 */
  'R', 0x00,                  /* wcChar1 */
  'M', 0x00,                  /* wcChar2 */
  ' ', 0x00,                  /* wcChar3 */
  'D', 0x00,                  /* wcChar4 */
  'e', 0x00,                  /* wcChar5 */
  'v', 0x00,                  /* wcChar6 */
  'i', 0x00,                  /* wcChar7 */
  'c', 0x00,                  /* wcChar8 */
  'e', 0x00,                  /* wcChar9 */
  ' ', 0x00,                  /* wcChar10 */
  'i', 0x00,                  /* wcChar11 */
  'n', 0x00,                  /* wcChar12 */
  ' ', 0x00,                  /* wcChar13 */
  'D', 0x00,                  /* wcChar14 */
  'F', 0x00,                  /* wcChar15 */
  'U', 0x00,                  /* wcChar16 */
  ' ', 0x00,                  /* wcChar17 */
  'M', 0x00,                  /* wcChar18 */
  'o', 0x00,                  /* wcChar19 */
  'd', 0x00,                  /* wcChar20 */
  'e', 0x00,                  /* wcChar21 */
  ///////////////////////////////////////
  /// string3 descriptor
  ///////////////////////////////////////
  0x16,                       /* bLength */
  USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
  '2', 0x00,                  /* wcChar0 */
  '0', 0x00,                  /* wcChar1 */
  '2', 0x00,                  /* wcChar2 */
  '4', 0x00,                  /* wcChar3 */
  '1', 0x00,                  /* wcChar4 */
  '0', 0x00,                  /* wcChar5 */
  '1', 0x00,                  /* wcChar6 */
  '5', 0x00,                  /* wcChar7 */
  '1', 0x00,                  /* wcChar8 */
  '0', 0x00,                  /* wcChar9 */
  #ifdef CONFIG_USB_HS
  ///////////////////////////////////////
  /// device qualifier descriptor
  ///////////////////////////////////////
  0x0a,
  USB_DESCRIPTOR_TYPE_DEVICE_QUALIFIER,
  0x00,
  0x02,
  0x00,
  0x00,
  0x00,
  0x40,
  0x01,
  0x00,
  #endif
  0x00
};

/*!< custom hid report descriptor */
static const uint8_t hid_custom_report_desc[HID_CUSTOM_REPORT_DESC_SIZE] =
{
  /* USER CODE BEGIN 0 */
  0x06, 0x00, 0xff, // USAGE_PAGE (Vendor Defined Page 1)
  0x09, 0x01,       // USAGE (Vendor Usage 1)
  0xa1, 0x01,       // COLLECTION (Application)
  0x09, 0x01,       //   USAGE (Vendor Usage 1)
  0x15, 0x00,       //   LOGICAL_MINIMUM (0)
  0x26, 0xff, 0x00, //   LOGICAL_MAXIMUM (255)
  0x95, 0x40,       //   REPORT_COUNT (64)
  0x75, 0x08,       //   REPORT_SIZE (8)
  0x81, 0x02,       //   INPUT (Data,Var,Abs)
  /* <___________________________________________________> */
  0x09, 0x01,       //   USAGE (Vendor Usage 1)
  0x15, 0x00,       //   LOGICAL_MINIMUM (0)
  0x26, 0xff, 0x00, //   LOGICAL_MAXIMUM (255)
  0x95, 0x40,       //   REPORT_COUNT (64)
  0x75, 0x08,       //   REPORT_SIZE (8)
  0x91, 0x02,       //   OUTPUT (Data,Var,Abs)
  /* USER CODE END 0 */
  0xC0 /*     END_COLLECTION               */
};

USB_NOCACHE_RAM_SECTION USB_MEM_ALIGNX uint8_t guc_usbd_read_buffer[64];

__IO uint8_t guc_usbd_hid_custom_out_callback_flag = 0;

#define HID_STATE_IDLE 0
#define HID_STATE_BUSY 1

/*!< hid state ! Data can be sent only when state is idle  */
static volatile uint8_t custom_state = HID_STATE_IDLE;

static void usbd_hid_custom_in_callback(uint8_t ep, uint32_t nbytes)
{
  USB_LOG_RAW("actual in len:%d\r\n", nbytes);
  custom_state = HID_STATE_IDLE;
}

static void usbd_hid_custom_out_callback(uint8_t ep, uint32_t nbytes)
{
  guc_usbd_hid_custom_out_callback_flag = 1;
  USB_LOG_RAW("actual out len:%d\r\n", nbytes);
  usbd_ep_start_read(HIDRAW_OUT_EP, guc_usbd_read_buffer, 64);
}

static struct usbd_endpoint custom_in_ep =
{
  .ep_cb = usbd_hid_custom_in_callback,
  .ep_addr = HIDRAW_IN_EP
};

static struct usbd_endpoint custom_out_ep =
{
  .ep_cb = usbd_hid_custom_out_callback,
  .ep_addr = HIDRAW_OUT_EP
};

struct usbd_interface intf0;

void usbd_event_handler(uint8_t event)
{
  switch (event)
  {
    case USBD_EVENT_RESET:
      break;

    case USBD_EVENT_CONNECTED:
      break;

    case USBD_EVENT_DISCONNECTED:
      break;

    case USBD_EVENT_RESUME:
      break;

    case USBD_EVENT_SUSPEND:
      break;

    case USBD_EVENT_CONFIGURED:
      usbd_ep_start_read(HIDRAW_OUT_EP, guc_usbd_read_buffer, 64);
      break;

    case USBD_EVENT_SET_REMOTE_WAKEUP:
      break;

    case USBD_EVENT_CLR_REMOTE_WAKEUP:
      break;

    default:
      break;
  }
}

void usb_dc_low_level_init(void)
{
  GPIO_InitTypeDef GPIO_InitStruct = {0};

  /* Peripheral clock enable */
  __HAL_RCC_USB1_CLK_ENABLE();
  __HAL_RCC_GPIOA_CLK_ENABLE();

  /**USART1 GPIO Configuration
  PA11    ------> USB1_DM
  PA12    ------> USB1_DP
  */
  GPIO_InitStruct.Pin       = GPIO_PIN_11 | GPIO_PIN_12;
  GPIO_InitStruct.Mode      = GPIO_MODE_AF_PP;
  GPIO_InitStruct.Pull      = GPIO_NOPULL;
  GPIO_InitStruct.Speed     = GPIO_SPEED_FREQ_HIGH;
  GPIO_InitStruct.Alternate = GPIO_AF14_USB1;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

  HAL_NVIC_SetPriorityGrouping(NVIC_PRIORITYGROUP_2);
  HAL_NVIC_SetPriority(OTG1_FS_IRQn, 2, 0);
  HAL_NVIC_EnableIRQ(OTG1_FS_IRQn);
}

/**
 * @brief  This function is used to configure USART pins and then initialize the used USART instance.
 * @retval None.
 */
static void OPENBL_USB_Init(void)
{
  OPENBL_WWDG_Refresh();
  OPENBL_IWDG_Refresh();

  usbd_desc_register(hid_descriptor);
  usbd_add_interface(usbd_hid_init_intf(&intf0, hid_custom_report_desc, HID_CUSTOM_REPORT_DESC_SIZE));
  usbd_add_endpoint(&custom_in_ep);
  usbd_add_endpoint(&custom_out_ep);

  usbd_initialize();
}

static void OPENBL_USB_DeInit(void)
{
  __HAL_RCC_USB1_CLK_DISABLE();
  __HAL_RCC_USB1_FORCE_RESET();
  __HAL_RCC_USB1_RELEASE_RESET();
  HAL_GPIO_DeInit(GPIOA, GPIO_PIN_11 | GPIO_PIN_12);
  HAL_NVIC_DisableIRQ(OTG1_FS_IRQn);
}

static void OPENBL_USB_Read(uint8_t *buf, uint16_t size)
{
  uint16_t tmpsize = 0;

  while (size)
  {
    tmpsize = size >= 64 ? 64 : size;

    while (0 == guc_usbd_hid_custom_out_callback_flag)
    {
      OPENBL_WWDG_Refresh();
      OPENBL_IWDG_Refresh();
    }

    guc_usbd_hid_custom_out_callback_flag = 0;

    memcpy(buf, guc_usbd_read_buffer, tmpsize);

    size -= tmpsize;
    buf += tmpsize;
  }
}

static void OPENBL_USB_Write(uint8_t *buf, uint16_t size)
{
  uint8_t sendbuffer[64];

  while (size)
  {
    if (size >= 64)
    {
      usbd_ep_start_write(HIDRAW_IN_EP, buf, 64);

      custom_state = HID_STATE_BUSY;

      while (HID_STATE_BUSY == custom_state)
      {
        OPENBL_WWDG_Refresh();
        OPENBL_IWDG_Refresh();
      }

      buf += 64;
      size -= 64;
    }
    else
    {
      memset(sendbuffer, 0x00, 64);
      memcpy(sendbuffer, buf, size);

      usbd_ep_start_write(HIDRAW_IN_EP, sendbuffer, 64);

      custom_state = HID_STATE_BUSY;

      while (HID_STATE_BUSY == custom_state)
      {
        OPENBL_WWDG_Refresh();
        OPENBL_IWDG_Refresh();
      }

      size = 0;
    }
  }
}

static uint8_t OPENBL_USB_ProtocolDetection(void)
{
  OPENBL_WWDG_Refresh();
  OPENBL_IWDG_Refresh();

  if (0 == guc_usbd_hid_custom_out_callback_flag)
  {
    return 0;
  }

  guc_usbd_hid_custom_out_callback_flag = 0;

  for (int i = 0; i < 64; i++)
  {
    if (0x7F != guc_usbd_read_buffer[i])
    {
      return 0;
    }
  }

  /* Aknowledge the host */
  uint8_t tmp = ACK_BYTE;
  OPENBL_USB_Write(&tmp, 1);

  return 1;
}

static const OPENBL_OpsTypeDef USB_Ops =
{
  .Init = OPENBL_USB_Init,
  .DeInit = OPENBL_USB_DeInit,
  .Detection = OPENBL_USB_ProtocolDetection,
  .CommandProcess = OPENBL_USART_CommandProcess,
};

static const OPENBL_IOTypeDef USB_IO =
{
  .Read = OPENBL_USB_Read,
  .Write = OPENBL_USB_Write,
};

OPENBL_HandleTypeDef USB1_Handle =
{
  .p_Ops = &USB_Ops,
  .p_IO = &USB_IO,
};

#endif /* IO_INTERFACE_USB1 */

/************************ (C) COPYRIGHT Puya *****END OF FILE****/
